/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbrecordset.h"
#include <iostream.h>

	/**
 	 * Constructor
   */		
	DBRecordSet::DBRecordSet()
	{
		m_poResult = 0;					// null pointer
		m_nCurrentRow = -1;			// point to before index
	} // end constructor
	
	/**
 	 * Destructor
   */		
	DBRecordSet::~DBRecordSet()
	{
		close();
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBRecordSet::isEmpty()
	{
		if (m_poResult == 0)
		{
			return true;
		} // end if result object is null
		if (PQntuples(m_poResult) == 0)
		{
			return true;
		} // end if no results
		return false;
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBRecordSet::getCurrentIndex()
	{
		return m_nCurrentRow;
	} // end getCurrentIndex
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBRecordSet::next()
	{
		if (isEmpty())
		{
			return false;
		} // if empty
		
		// check if max has been reached
		if ((m_nCurrentRow + 1) >= getRecordCount())
		{
			return false;
		} // end if max reached

		// move to next record
		m_nCurrentRow++;

		// DEBUG
				
		return true;
		
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBRecordSet::previous()
	{
		if (isEmpty())
		{
			return false;
		} // end if empty	
		
		// check if min has been reached
		if (m_nCurrentRow <= 0)
		{
			return false;
		} // end if minimum reached
		// move to previous record
		m_nCurrentRow--;
		return true;
	} // end previous

	/**
	 * Used to get the value of a field from the current record.
	 * @param		nFieldIndex	the field index starting from 0
	 * @return	the field value as a string
	 */
	const string DBRecordSet::getFieldValue(int nFieldIndex) throw (NoRecordException, InvalidFieldException)
	{				
		char *pcFieldValue = 0;
		// check that record exists
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot retrieve field value for empty record set."), string("DBRecordSet"), string("getFieldValue"));
		} // end if empty			
		// check field index is valid
		if (nFieldIndex < 0 || nFieldIndex >= getFieldCount())
		{
			throw InvalidFieldException(string("The specified field index is not within the available range."), string("DBRecordSet"), string("getFieldValue"));		
		} // end if field index is invalid
		
		pcFieldValue = PQgetvalue(m_poResult, m_nCurrentRow, nFieldIndex);
		return string(pcFieldValue);		
	} // end getFieldValue

	/**
	 * Used to get the value of a field from the current record.
	 * @param		rstrFieldName	the field name
	 * @return	the field value as a string
	 */
	const string DBRecordSet::getFieldValue(const string & rstrFieldName) throw (NoRecordException, InvalidFieldException)
	{
		int nFieldIndex = 0;
		string strFieldValue;
		// check that record exists
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot retrieve field value for empty record set."), string("DBRecordSet"), string("getFieldValue"));
		} // end if empty			
		nFieldIndex = PQfnumber(m_poResult, rstrFieldName.c_str());
		// check field is valid
		if (nFieldIndex == -1)
		{
			throw InvalidFieldException(string("The specified field name does not exist."), string("DBRecordSet"), string("getFieldValue"));		
		} // end if
		
		// get value
		return getFieldValue(nFieldIndex);
	} // end getFieldValue

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBRecordSet::reset()
	{
		m_nCurrentRow = -1;			// point to before index
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBRecordSet::first() throw (NoRecordException)
	{
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot move to first record, no records exist."), string("DBRecordSet"), string("first"));
		} // if empty
		
		// move to first record
		m_nCurrentRow = 0;	
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */

	void DBRecordSet::last() throw (NoRecordException)
	{
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot move to last record, no records exist."), string("DBRecordSet"), string("first"));
		} // if empty
		
		// move to last record
		m_nCurrentRow = getRecordCount() - 1;		
	} // end last

	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBRecordSet::getRecordCount()		
	{
		if (isEmpty())
		{
			return 0;
		} // end if is empty
		return PQntuples(m_poResult);
	} // end getRecordCount

	/**
	 * Used to get the number of fields in this record set.
	 * @return 	the number of fields
	 */
	int DBRecordSet::getFieldCount()
	{
		if (isEmpty())
		{
			return 0;
		} // end if is empty
		return PQnfields(m_poResult);				
	} // end getFieldCount
	
	/**
	 * Used to identify if a field value is null in the current record.
	 * @param		nFieldIndex the field index starting from 0
	 * @return	true if the field value is null, false otherwise
	 */
	bool DBRecordSet::isFieldNull(int nFieldIndex) throw (NoRecordException, InvalidFieldException)
	{				
		// check that record exists
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot retrieve field value for empty record set."), string("DBRecordSet"), string("isFieldNull"));
		} // end if empty			
		// check field index is valid
		if (nFieldIndex < 0 || nFieldIndex >= getFieldCount())
		{
			throw InvalidFieldException(string("The specified field index is not within the available range."), string("DBRecordSet"), string("isFieldNull"));		
		} // end if field index is invalid
		
		return PQgetisnull(m_poResult, m_nCurrentRow, nFieldIndex);
	} // end isFieldNull
	
	/**
	 * Used to identify if a field value is null in the current record.
	 * @param		rstrFieldName the field index starting from 0
	 * @return	true if the field value is null, false otherwise
	 */
	bool DBRecordSet::isFieldNull(const string & rstrFieldName) throw (NoRecordException, InvalidFieldException)
	{	
		int nFieldIndex = 0;
		// check that record exists
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot retrieve field value for empty record set."), string("DBRecordSet"), string("isFieldNull"));
		} // end if empty			
		nFieldIndex = PQfnumber(m_poResult, rstrFieldName.c_str());
		// check field is valid
		if (nFieldIndex == -1)
		{
			throw InvalidFieldException(string("The specified field name does not exist."), string("DBRecordSet"), string("isFieldNull"));		
		} // end if
		
		return PQgetisnull(m_poResult, m_nCurrentRow, nFieldIndex);
	} // end isFieldNull
	

	/**
 	 * Used to free resources associated with the record set.
   */			    			
	void DBRecordSet::close()
	{
		// check if anything to clean up
		if (m_poResult != 0)
		{
			PQclear(m_poResult);
			m_poResult = 0;	// set null			
		} // end if not null
	} // end close

