/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbindexset.h"
#include "../../dbconnection.h"
#include "../../../utils/debugger.h"

	/**
 	 * Constructor 	
   */		
  DBIndexSet::DBIndexSet()
  {
  	// set to null pointer
  	m_aoIndexedColumns = 0;
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBIndexSet::~DBIndexSet()
	{
		// clean up array of recordsets if need be
		if (m_aoIndexedColumns != 0)
		{
			delete [] m_aoIndexedColumns;
		} // end if array set		
	} // end destructor

	/**
	 * Used to indicate if the record set is empty.
	 * @return	true if record set is empty, false otherwise
	 */
	bool DBIndexSet::isEmpty()
	{
		return m_oIndexList.isEmpty();
	} // end isEmpty

	/**
	 * Used to get the current index location.
	 * @return	the current index
	 */
	int DBIndexSet::getCurrentIndex()
	{
		return m_oIndexList.getCurrentIndex();
	} // end getCurrentIndex
		
	/**
	 * Used to move to the next record in the record set. This function must be called before
	 * the first result can be retrieved. When a record set is initially set, the internal
	 * pointer points to a location before the first record. The reason for this is so that
	 * the record set may sit in a while loop with calls to next indicating if more records
	 * are to come. The function returns true when more records exist and false when no more
	 * records are to come.
	 *
	 * @return	true when more records to come, false when no more records
	 */
	bool DBIndexSet::next()
	{
		return m_oIndexList.next();	
	} // end next

	/**
	 * Used to move to the previous record in the record set. The function returns true when
	 * a previous record exists, it will return false when the begining of the set is reached.
	 *
	 * @return	true when previous record exists, false when first record reached (or when empty record set)
	 */
	bool DBIndexSet::previous()
	{
		return m_oIndexList.previous();
	} // end previous

	/**
	 * Used to move the cursor back to a position before the first record. This is used when
	 * the recordset needs to be used but then reset, so that next() function can be used
	 * correctly.
	 */
	void DBIndexSet::reset()
	{
		m_oIndexList.reset();
	} // end reset
	
	/**
	 * Used to move to the first record.
	 * @exception	throws NoRecordException when empty record
	 */
	void DBIndexSet::first() throw (NoRecordException)
	{
		return m_oIndexList.first();
	} // end first

	/**
	 * Used to move to the last record.
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBIndexSet::last() throw (NoRecordException)
	{
		return m_oIndexList.last();	
	} // end last
				
	/**
	 * Used to get the number of records in this record set.
	 * @return 	the number of records
	 */
	int DBIndexSet::getRecordCount()
	{
		return m_oIndexList.getRecordCount();		
	} // end getRecordCount

	/**
	 * Used to get the current index.
	 * @param	roDBIndex a DBIndex object used to return output for current record
	 * @exception	throws NoRecordException when empty record	
	 */
	void DBIndexSet::getDBIndex(DBIndex & roDBIndex) throw (NoRecordException)
	{
		string strMethodName = "DBIndexSet::getDBIndex";
		
		Debugger::entered(strMethodName);
		
		if (isEmpty())
		{
			throw NoRecordException(string("Cannot retrieve DBIndex object, no records exist."), string("DBIndexSet"), string("getDBIndex"));
		} // end if empty
		try
		{
			roDBIndex.setTableName(m_oIndexList.getFieldValue("tablename"));
			roDBIndex.setIndexName(m_oIndexList.getFieldValue("indexname"));			
			roDBIndex.setIndexComment(m_oIndexList.getFieldValue("comment"));						
			if (m_oIndexList.getFieldValue("isunique") == DBConnection::DB_TRUE)
			{
				roDBIndex.setUnique(true);						
			} // end if unique
			else
			{
				roDBIndex.setUnique(false);									
			} // end else not unique
			if (m_oIndexList.getFieldValue("isprimary") == DBConnection::DB_TRUE)
			{			
				roDBIndex.setPrimary(true);						
			} // end if primary
			else
			{
				roDBIndex.setPrimary(false);									
			} // end else not primary

			// set indexed column names
			while (m_aoIndexedColumns[getCurrentIndex()].next())
			{
				roDBIndex.addIndexedColumn(m_aoIndexedColumns[getCurrentIndex()].getFieldValue(string("column")));
			} // end while more columns
		} // end try
		catch (VPException e)
		{
			// debug this
			Debugger::logException(strMethodName, e);	
		} // end catch	
		
		Debugger::exited(strMethodName);	
	} // end getDBIndex

