/* Module:          SQLPutData.c
 *
 * Description:     Allows an application to send data for a parameter or 
 *					column to the driver at statement execution time. 
 *
 * Classes:         
 *
 * API functions:   SQLPutData
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLPutData(
							 SQLHSTMT		hDrvStmt,
							 SQLPOINTER		pData,
							 SQLINTEGER		nLengthOrIndicator
							 )
{
	static char *func = "SQLPutData";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	int old_pos, retval;
	ParameterInfoClass *current_param;
	char *buffer;

	mylog( "%s: entering...\n", func);

	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	
	if (stmt->current_exec_param < 0)
	{
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		stmt->errormsg = "Previous call was not SQLPutData or SQLParamData";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	current_param = &(stmt->parameters[stmt->current_exec_param]);

	if ( ! stmt->put_data)
	{	/* first call */

		mylog("SQLPutData: (1) nLengthOrIndicator = %d\n", nLengthOrIndicator);

		stmt->put_data = TRUE;

		current_param->EXEC_used = (SQLINTEGER *) malloc(sizeof(SQLINTEGER));
		if ( ! current_param->EXEC_used)
		{
			stmt->errornumber = STMT_NO_MEMORY_ERROR;
			stmt->errormsg = "Out of memory in SQLPutData (1)";
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		*current_param->EXEC_used = nLengthOrIndicator;

		if (nLengthOrIndicator == SQL_NULL_DATA)
		{
			return SQL_SUCCESS;
		}


		/*	Handle Long Var Binary with Large Objects */
		if ( current_param->SQLType == SQL_LONGVARBINARY)
		{

			/* begin transaction if needed */
			if(!CC_is_in_trans(stmt->hdbc))
			{
				QResultClass *res;
				char ok;

				res = CC_send_query(stmt->hdbc, "BEGIN", NULL);
				if (!res)
				{
					stmt->errormsg = "Could not begin (in-line) a transaction";
					stmt->errornumber = STMT_EXEC_ERROR;
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}
				ok = QR_command_successful(res);
				QR_Destructor(res);
				if (!ok)
				{
					stmt->errormsg = "Could not begin (in-line) a transaction";
					stmt->errornumber = STMT_EXEC_ERROR;
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}

				CC_set_in_trans(stmt->hdbc);
			}

			/*	store the oid */
			current_param->lobj_oid = lo_creat(stmt->hdbc, INV_READ | INV_WRITE);
			if (current_param->lobj_oid == 0)
			{
				stmt->errornumber = STMT_EXEC_ERROR;
				stmt->errormsg = "Couldnt create large object.";
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}

			/*	major hack -- to allow convert to see somethings there */
			/*					have to modify convert to handle this better */
			current_param->EXEC_buffer = (char *) &current_param->lobj_oid;

			/*	store the fd */
			stmt->lobj_fd = lo_open(stmt->hdbc, current_param->lobj_oid, INV_WRITE);
			if ( stmt->lobj_fd < 0)
			{
				stmt->errornumber = STMT_EXEC_ERROR;
				stmt->errormsg = "Couldnt open large object for writing.";
				SC_log_error(func, "", stmt);
				return SQL_ERROR;
			}

			retval = lo_write(stmt->hdbc, stmt->lobj_fd, pData, nLengthOrIndicator);
			mylog("lo_write: nLengthOrIndicator=%d, wrote %d bytes\n", nLengthOrIndicator, retval);

		}
		else 
		{	/* for handling text fields and small binaries */

			if (nLengthOrIndicator == SQL_NTS)
			{
				current_param->EXEC_buffer = strdup(pData);
				if ( ! current_param->EXEC_buffer)
				{
					stmt->errornumber = STMT_NO_MEMORY_ERROR;
					stmt->errormsg = "Out of memory in SQLPutData (2)";
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}
			}
			else
			{
				current_param->EXEC_buffer = malloc(nLengthOrIndicator + 1);
				if ( ! current_param->EXEC_buffer)
				{
					stmt->errornumber = STMT_NO_MEMORY_ERROR;
					stmt->errormsg = "Out of memory in SQLPutData (2)";
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}
				memcpy(current_param->EXEC_buffer, pData, nLengthOrIndicator);
				current_param->EXEC_buffer[nLengthOrIndicator] = '\0';
			}
		}
	}

	else
	{	/* calling SQLPutData more than once */

		mylog("SQLPutData: (>1) nLengthOrIndicator = %d\n", nLengthOrIndicator);

		if (current_param->SQLType == SQL_LONGVARBINARY)
		{

			/* the large object fd is in EXEC_buffer */
			retval = lo_write(stmt->hdbc, stmt->lobj_fd, pData, nLengthOrIndicator);
			mylog("lo_write(2): nLengthOrIndicator = %d, wrote %d bytes\n", nLengthOrIndicator, retval);

			*current_param->EXEC_used += nLengthOrIndicator;

		}
		else
		{

			buffer = current_param->EXEC_buffer;

			if (nLengthOrIndicator == SQL_NTS)
			{
				buffer = realloc(buffer, strlen(buffer) + strlen(pData) + 1);
				if ( ! buffer)
				{
					stmt->errornumber = STMT_NO_MEMORY_ERROR;
					stmt->errormsg = "Out of memory in SQLPutData (3)";
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}
				strcat(buffer, pData);

				mylog("       nLengthOrIndicator = SQL_NTS: strlen(buffer) = %d\n", strlen(buffer));

				*current_param->EXEC_used = nLengthOrIndicator;

				/*	reassign buffer incase realloc moved it */
				current_param->EXEC_buffer = buffer;

			}
			else if (nLengthOrIndicator > 0)
			{

				old_pos = *current_param->EXEC_used;

				*current_param->EXEC_used += nLengthOrIndicator;

				mylog("        nLengthOrIndicator = %d, old_pos = %d, *used = %d\n", nLengthOrIndicator, old_pos, *current_param->EXEC_used);

				/* dont lose the old pointer in case out of memory */
				buffer = realloc(current_param->EXEC_buffer, *current_param->EXEC_used + 1);
				if ( ! buffer)
				{
					stmt->errornumber = STMT_NO_MEMORY_ERROR;
					stmt->errormsg = "Out of memory in SQLPutData (3)";
					SC_log_error(func, "", stmt);
					return SQL_ERROR;
				}

				memcpy(&buffer[old_pos], pData, nLengthOrIndicator);
				buffer[*current_param->EXEC_used] = '\0';

				/*	reassign buffer incase realloc moved it */
				current_param->EXEC_buffer = buffer;
				
			}
			else
			{
				SC_log_error(func, "bad nLengthOrIndicator", stmt);
				return SQL_ERROR;
			}

		}
	}


	return SQL_SUCCESS;
}