#!/usr/local/bin/ruby

require "test/unit"
require "pqa.rb"

class QueryTest < Test::Unit::TestCase
	def test_add
		q = Query.new
		q.append("test")
		assert(q.to_s == " test", "append should insert leading space, instead:" + q.to_s + ":")	
	end
	def test_normalize_names
		assert(Query.new("WHERE unix_group_name='wtr'").normalize.to_s == "WHERE unix_group_name=''", "normalize should remove stuff in quotes")
	end
	def test_normalize_spaces
		assert(Query.new("WHERE    unix_group_name='wtr'").normalize.to_s == "WHERE unix_group_name=''", "normalize should squash spaces")
	end
	def test_eql_hash
		a = [Query.new("test test test"), Query.new("test test test")]
		assert(a.uniq.size == 1, "eql/hash not working")
	end
	def test_is_select
		assert(Query.new("SELECT * from users").is_select, "SELECT detection failed")
		assert(Query.new("select * from users").is_select, "SELECT detection failed")
	end
	def test_is_delete
		assert(Query.new("DELETE * from users"), "DELETE detection failed")
	end
	def test_is_insert
		assert(Query.new("insert into users values etc etc"), "INSERT detection failed")
	end
end

class SyslogLineTest < Test::Unit::TestCase
	def test_is_pg_log_entry
		assert(SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755] LOG:  query: SELECT * FROM plugins").is_pg_log_entry, "Didn't recognize legit PG log entry")
		assert(!SyslogLine.new("Mar  9 14:29:05 hal modprobe: modprobe: Can't locate module sound-service-1-0").is_pg_log_entry, "False positive")
	end
	def test_connection_id_no_dash
		assert(SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755] LOG:  query: SELECT * FROM plugins").connection_id == "2755", "Couldn't parse connection id")
	end
	def test_connection_id_with_dash
		assert(SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755-1] LOG:  query: SELECT * FROM plugins").connection_id == "2755", "Couldn't parse connection id")
	end
	def test_is_new_query	
		assert(SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755] LOG:  query: SELECT * FROM plugins").is_new_query, "Didn't identify new query")
	end
	def test_parse_query_all
		txt = SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2755] LOG:  query: SELECT * FROM plugins").parse_query_segment
		assert(txt ="SELECT * FROM plugins", "Didn't parse query segment correctly:" + txt + ":")
	end
	def test_parse_query_partial
		txt = SyslogLine.new("Feb 25 04:03:00 rubyforge postgres[4545]: [2756-3] ^I^I^IFROM groups").parse_query_segment
		assert(txt ="FROM groups", "Didn't parse query segment correctly:" + txt + ":")
	end
end

class PostgresLogLineTest < Test::Unit::TestCase
	ENTRY_WITH_DATE = "2004-04-27 10:54:18 LOG:  query: SELECT language_code FROM supported_languages WHERE language_id='1'"
	ENTRY_WITHOUT_DATE = "LOG:  query: SELECT language_code FROM supported_languages WHERE language_id='1'"
	ENTRY_NON_QUERY = "postmaster successfully started"
	ENTRY_QUERY_CONTINUATION = "\t\tfrom users"
	def test_snip_date
		line = PostgresLogLine.new(ENTRY_WITH_DATE)
		assert(/^LOG/.match(line.text).size == 1, "Didn't snip date correctly")
		line = PostgresLogLine.new(ENTRY_WITHOUT_DATE)
		assert(/^LOG/.match(line.text).size == 1, "Failed to process line without date")
	end
	def test_is_new_query
		line = PostgresLogLine.new(ENTRY_WITH_DATE)
		assert(line.is_new_query, "Failed to identify new query")
		line = PostgresLogLine.new(ENTRY_NON_QUERY)
		assert(!line.is_new_query, "Failed to identify non-query-line query")
	end
	def test_query_continuation
		line = PostgresLogLine.new(ENTRY_QUERY_CONTINUATION)
		assert(!line.is_new_query, "False positive on a query continuation")
		assert(line.is_continuation, "Failed to identify query continuation")
	end
	def test_parse_segment_from_new_query
		line = PostgresLogLine.new(ENTRY_WITHOUT_DATE)
		assert(line.parse_query_segment == "SELECT language_code FROM supported_languages WHERE language_id='1'", "Failed to parse query segment: #{line.parse_query_segment}")
	end
	def test_parse_segment_from_continuation
		line = PostgresLogLine.new(ENTRY_QUERY_CONTINUATION)
		assert(line.parse_query_segment == "from users", "Failed to parse query segment from continuation")
	end
end

class AccumulatorTest < Test::Unit::TestCase
	def test_new_line
		a = Accumulator.new
		a.new_query_start("select * from")
		a.query_continuation("users")
		#a.create_query
	end	
end
