/* $Id: portal.c,v 1.13 2005/12/14 19:25:54 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres Portal interface
 */
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <executor/executor.h>
#include <executor/execdesc.h>
#include <executor/tstoreReceiver.h>
#include <nodes/pg_list.h>
#include <nodes/params.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <tcop/pquery.h>
#include <tcop/utility.h>
#include <utils/array.h>
#include <utils/palloc.h>
#include <utils/memutils.h>
#include <utils/portal.h>
#include <utils/relcache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/error.h>

#include <pypg/query.h>
#include <pypg/call.h>
#include <pypg/call/portal.h>

static void
_R_Receive
(
#if PGV_MM == 80
 	HeapTuple ht, TupleDesc td,
#else
	TupleTableSlot *slot,
#endif
	DestReceiver *r
)
{
	PyObj port, tup, tupl;
#if PGV_MM != 80
	HeapTuple ht = ExecFetchSlotTuple(slot);
#endif

	port = (PyObj) ((void *) r - (void *) offsetof(struct PyPgPortal, p_rec));

	tup = PyPgHeapTuple_New(PyPgCall_FetchOutput(port), ht);
	tupl = PyPgCall_FetchReturned(port);
	if (tupl == Py_None)
	{
		Py_DECREF(Py_None);
		tupl = NULL;
	}

	if (tupl)
		PyList_Append(tupl, tup);
	else
		PyPgCall_FixReturned(port, tup);
}
static void _R_Startup(DestReceiver *r, int op, TupleDesc td) {}
static void _R_Shutdown(DestReceiver *r) {}
static void _R_Destroy(DestReceiver *r) {}

static void
_R_InitReceiver(DestReceiver *r)
{
#if PGV_MM == 80
	r->receiveTuple
#else
	r->receiveSlot
#endif
							= _R_Receive;
	r->rStartup			= _R_Startup;
	r->rShutdown		= _R_Shutdown;
	r->rDestroy			= _R_Destroy;
	r->mydest			= 0xFFA0;
}

static PyMemberDef PyPgPortal_Members[] = {
	{"query", T_OBJECT, offsetof(struct PyPgPortal, call_obj), RO,
	"the query from which the Postgres.Portal was created"},
	{"name", T_OBJECT, offsetof(struct PyPgPortal, p_name), RO,
	"the name of the Portal that this Postgres.Portal created"},
	{NULL}
};

static void
port_closedError(PyObj self)
{
	/* Portal p = PyPgPortal_FetchPortal(self); */
	PyErr_Format(PyExc_ValueError, "operation on closed Postgres.Portal");
}
#define RaiseClosedPortalError(SELF, RET) do { \
	if (PyPgPortal_IsClosed(SELF)) { \
		port_closedError(SELF); \
		return(RET); \
	} \
} while(0)

static PyObj
_port_read(PyObj self, long q)
{
	volatile PyObj rob = NULL;
	Portal p;
	DestReceiver *r;
	p = PyPgPortal_FetchPortal(self);
	r = PyPgPortal_FetchReceiver(self);
	if (q != 0)
		PgError_TRAP(PortalRunFetch(p, FETCH_FORWARD, q, r));

	if (!PyErr_Occurred())
	{
		rob = PyPgCall_FetchReturned(self);
		PyPgCall_FixReturned(self, Py_None);
		Py_INCREF(Py_None);
	}
	return(rob);
}

static PyObj
port_read(PyObj self, PyObj args)
{
	PyObj quantity = NULL, rob = NULL;
	long q;

	RaiseClosedPortalError(self, NULL);
	if (!PyArg_ParseTuple(args, "|O", &quantity))
		return(NULL);
	
	if (quantity != Py_None && quantity != NULL)
	{
		PyObj intobj;
		intobj = PyNumber_Int(quantity);
		if (intobj == NULL) return(NULL);
		q = PyInt_AsLong(quantity);
	}
	else
		q = FETCH_ALL;

	if (PyPgPortal_AtEnd(self) && q > 0)
		return(PyList_New(0));
	else if (PyPgPortal_AtGenesis(self) && q < 0)
		return(PyList_New(0));

	PyPgCall_FixReturned(self, PyList_New(0));
	rob = _port_read(self, q);
	return(rob);
}

static void
_port_seek(PyObj self, long offset, FetchDirection fdir)
{
	Portal p = PyPgPortal_FetchPortal(self);
	PortalRunFetch(p, fdir, offset, None_Receiver);
}

static PyObj
port_seek(PyObj self, PyObj args, PyObj kw)
{
	static char *words[] = {"offset", "whence", NULL};
	long offset, pos, whence = 0;
	FetchDirection fdir;

	RaiseClosedPortalError(self, NULL);
	if (!PyArg_ParseTupleAndKeywords(args, kw, "l|l", words, &offset, &whence))
		return(NULL);

	pos = PyPgPortal_FetchPosition(self);
	if (whence == 0)
		fdir = FETCH_ABSOLUTE;
	else if (whence == 1)
		fdir = FETCH_RELATIVE;
	else if (whence == 2)
	{
		_port_seek(self, FETCH_ALL, FETCH_ABSOLUTE);
		fdir = FETCH_RELATIVE;
		offset = -offset - 1;
	}
	else
	{
		PyErr_Format(PyExc_ValueError,
			"invalid whence, %ld, for Portal.seek", whence);
		return(NULL);
	}

	_port_seek(self, offset, fdir);
	return(PyInt_FromLong(PyPgPortal_FetchPosition(self) - pos));
}

static PyObj
port_scroll(PyObj self, PyObj args)
{
	long position;

	RaiseClosedPortalError(self, NULL);
	if (!PyArg_ParseTuple(args, "l", &position))
		return(NULL);

	_port_seek(self, position, FETCH_RELATIVE);
	RETURN_NONE;
}

static PyObj
port_tell(PyObj self)
{
	PyObj rob;
	RaiseClosedPortalError(self, NULL);
	rob = PyInt_FromLong(PyPgPortal_FetchPosition(self));
	return(rob);
}

static PyObj
port_close(PyObj self)
{
	if (!PyPgPortal_IsClosed(self))
	{
		Portal p = PyPgPortal_FetchPortal(self);
		PyPgPortal_FixPortal(self, NULL);
		PortalDrop(p, false);
	}

	RETURN_NONE;
}

static PyMethodDef PyPgPortal_Methods[] = {
	{"read", (PyCFunction) port_read, METH_VARARGS,
		"read the given number of tuples from the portal"},
	{"seek", (PyCFunction) port_seek, METH_VARARGS|METH_KEYWORDS,
		"seek to the given tuple offset of the portal"},
	{"scroll", (PyCFunction) port_scroll, METH_VARARGS,
		"scroll n-tuples relative to the current position"},
	{"tell", (PyCFunction) port_tell, METH_NOARGS,
		"return the current tuple offset"},
	{"close", (PyCFunction) port_close, METH_NOARGS,
		"close the portal to prevent further use and to free up resources"},
	{NULL}
};

static int
port_length(PyObj self)
{
	long len, state;
	state = PyPgPortal_FetchPosition(self);
	_port_seek(self, FETCH_ALL, FETCH_ABSOLUTE);
	len = PyPgPortal_FetchPosition(self);
	_port_seek(self, state, FETCH_ABSOLUTE);
	return(len);
}

static PyObj
port_item(PyObj self, int index)
{
	long curpos = PyPgPortal_FetchPosition(self);
	PyObj rob = NULL;

	_port_seek(self, index, FETCH_ABSOLUTE);
	if (PyPgPortal_FetchPosition(self) < index)
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.Portal index, \"%d\", out of range", index);
	}
	else
		rob = _port_read(self, 1);

	_port_seek(self, curpos, FETCH_ABSOLUTE);
	return(rob);
}

static PyObj
port_slice(PyObj self, int from, int to)
{
	long curpos = PyPgPortal_FetchPosition(self);
	PyObj rob = NULL;

	_port_seek(self, from, FETCH_ABSOLUTE);
	if (PyPgPortal_FetchPosition(self) != from)
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.Portal index, \"%d\", out of range", from);
	}
	else
	{
		PyPgCall_FixReturned(self, PyList_New(0));
		rob = _port_read(self, to - from);
	}

	_port_seek(self, curpos, FETCH_ABSOLUTE);
	return(rob);
}

static PySequenceMethods PyPgPortalAsSequence = {
	port_length,	/* sq_length */
	NULL,				/* sq_concat */
	NULL,				/* sq_repeat */
	port_item,		/* sq_item */
	port_slice,		/* sq_slice */
	NULL,				/* sq_ass_item */
};

static PyObj
port_iternext(PyObj self)
{
	PyObj rob = NULL;

	RaiseClosedPortalError(self, NULL);

	if (!PyPgPortal_AtEnd(self))
	{
		rob = _port_read(self, 1);
		if (rob == Py_None)
		{
			rob = NULL;
			Py_DECREF(Py_None);
		}
	}
	return(rob);
}

static void
port_dealloc(PyObj self)
{
	PyObj ob;

	if (!PyPgPortal_IsClosed(self))
		port_close(self);

	ob = PyPgPortal_FetchName(self);
	Py_DECREF(ob);
	PyPgPortal_FixName(self, NULL);

	self->ob_type->tp_base->tp_dealloc(self);
}

static PyObj
port_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob;
	PyErr_Warn(PyExc_DeprecationWarning, "use the next() method instead");
	rob = self->ob_type->tp_iternext(self);
	return(rob);
}

static PyObj
port_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source = NULL, source_str, output, rob = NULL;
	Portal port = NULL;

	if (!PyPgArg_ParseTypeSource(args, kw, &source))
		return(NULL);

	source_str = PyObject_Str(source);
	if (source_str == NULL) return(NULL);

	PG_TRY();
	{
		port = GetPortalByName(PyString_AS_STRING(source_str));
		if (port == NULL)
			ereport(ERROR, (
				errcode(ERRCODE_INVALID_CURSOR_NAME),
				errmsg("cursor \"%s\" does not exist",
					PyString_AS_STRING(source_str))
			));
	}
	PYPG_CATCH_END();
	if (PyErr_Occurred()) goto DECREF_source_str;

	rob = subtype->tp_alloc(subtype, 0);
	if (rob == NULL) goto DECREF_source_str;

	PyPgPortal_FixName(rob, source_str);
	PyPgPortal_FixPortal(rob, port);

	output = PyPgTupleDesc_New(port->tupDesc);
	if (output == NULL) goto free_rob;
	PyPgCall_FixOutput(rob, output);

	PyPgCall_FixReturned(rob, Py_None);
	Py_INCREF(Py_None);

	PyPgCall_FixFunction(rob, Py_None);
	Py_INCREF(Py_None);
	PyPgCall_FixInput(rob, Py_None);
	Py_INCREF(Py_None);
	_R_InitReceiver(PyPgPortal_FetchReceiver(rob));

	return(rob);
free_rob:
	subtype->tp_free(rob);
DECREF_source_str:
	Py_DECREF(source_str);
	return(NULL);
}

PyDoc_STRVAR(PyPgPortal_Doc, "Postgres Portal");
PyTypeObject PyPgPortal_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Portal",				/* tp_name */
	sizeof(struct PyPgPortal),		/* tp_basicsize */
	0,										/* tp_itemsize */
	port_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	NULL,									/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgPortalAsSequence,			/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	port_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	PyPgPortal_Doc,					/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	(getiterfunc) Py_RETURN_SELF,	/* tp_iter */
	port_iternext,						/* tp_iternext */
	PyPgPortal_Methods,				/* tp_methods */
	PyPgPortal_Members,				/* tp_members */
	NULL,									/* tp_getset */
	&PyPgCall_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	port_new,							/* tp_new */
	NULL,
};

PyObj
PyPgPortal_Initialize(PyObj self, PyObj query, PyObj input)
{
	Portal port = NULL;
	PyObj tdo, pname;

	if (self == NULL) return(NULL);

	pname = PyString_FromFormat("<Python Portal %p>", self);
	if (pname == NULL) goto free_self;
	PyPgPortal_FixName(self, pname);

	PG_TRY();
	{
		HeapTuple ht;
		TupleDesc td;
		ParamListInfo pli;
		MemoryContext former;

		port = CreatePortal(PyString_AS_STRING(pname), true, true);

		ht = PyPgHeapTuple_FetchHeapTuple(input);
		td = PyPgHeapTuple_FetchTupleDesc(input);

		former = MemoryContextSwitchTo(PortalGetHeapMemory(port));
		pli = ParamListInfo_FromTupleDescAndHeapTuple(td, ht);
		MemoryContextSwitchTo(former);

		PortalDefineQuery(port,
			NULL,
			"SELECT",
			PyPgQuery_FetchQueryList(query),
			PyPgQuery_FetchPlanList(query),
			PortalGetHeapMemory(port)
		);

		port->cursorOptions = CURSOR_OPT_SCROLL | CURSOR_OPT_HOLD;
		PortalStart(port, pli, GetLatestSnapshot());
	}
	PG_CATCH();
	{
		if (port) PortalDrop(port, false);
		PyErr_SetPgError();
	}
	PG_END_TRY();

	if (PyErr_Occurred()) goto DECREF_pname;
	PyPgPortal_FixPortal(self, port);

	tdo = PyPgTupleDesc_New(port->tupDesc);
	if (tdo == NULL) goto DECREF_pname;
	PyPgCall_FixOutput(self, tdo);

	PyPgCall_FixReturned(self, Py_None);
	Py_INCREF(Py_None);

	PyPgCall_FixFunction(self, query);
	Py_INCREF(query);
	PyPgCall_FixInput(self, input);
	Py_INCREF(input);
	_R_InitReceiver(PyPgPortal_FetchReceiver(self));

	return(self);
DECREF_pname:
	Py_DECREF(pname);
free_self:
	self->ob_type->tp_free(self);
	return(NULL);
}
/*
 * vim: ts=3:sw=3:noet:
 */
